#!/bin/bash

# ==============================================================================
#      KubeVPN Manager - 全自动安装脚本 (直接下载版)
# ==============================================================================
#
# 这个脚本会自动完成所有安装步骤:
# 1. 直接从指定 URL 下载 KubeVPN_Manager 程序到 ~/.local/ 目录。
# 2. 授予程序执行权限。
# 3. 创建一个 .desktop 快捷方式，使用您最终确认的方案。
#
# ==============================================================================

# --- 配置部分 ---
APP_NAME="KubeVPN Manager"
APP_EXECUTABLE_NAME="KubeVPN_Manager"
DOWNLOAD_URL="https://file.devops.biaoguoworks.com/logfile/KubeVPN_Manager"
APP_DIR="$HOME/.local"
APP_PATH="$APP_DIR/$APP_EXECUTABLE_NAME"
DESKTOP_FILENAME="$APP_NAME.desktop"

# --- 脚本开始 ---

echo "--- 正在全自动安装 $APP_NAME ---"

# 1. 确保目标目录存在
mkdir -p "$APP_DIR"
echo "➡️  [1/4] 确保安装目录存在: $APP_DIR"

# 2. 检查并下载程序（如果需要）
echo "➡️  [2/4] 检查主程序是否存在..."
if [ -f "$APP_PATH" ]; then
    echo "   -> ✅ '$APP_EXECUTABLE_NAME' 已存在，跳过下载。"
else
    echo "   -> 程序不存在，开始从 URL 下载..."
    # 直接下载程序到目标目录
    curl -L -s -o "$APP_PATH" "$DOWNLOAD_URL"

    if [ $? -ne 0 ] || [ ! -s "$APP_PATH" ]; then
        echo "   -> ❌ 错误：下载文件失败！请检查网络连接或 URL 是否正确。"
        rm -f "$APP_PATH" # 清理可能创建的空文件
        exit 1
    fi
    echo "   -> ✅ 下载完成。"
fi

# 3. 授予程序可执行权限
echo "➡️  [3/4] 正在授予程序执行权限..."
chmod +x "$APP_PATH"
echo "✅ 授权成功。"

# 4. 创建 .desktop 快捷方式
echo "➡️  [4/4] 正在创建桌面快捷方式..."
APP_MENU_PATH="$HOME/.local/share/applications"
mkdir -p "$APP_MENU_PATH"

# 使用 cat 和 <<'EOF' 的可靠方式创建文件
cat <<EOF > "$APP_MENU_PATH/$DESKTOP_FILENAME"
[Desktop Entry]
Version=1.0
Type=Application
Name=$APP_NAME (Admin)
Comment=以管理员身份启动 $APP_NAME
Exec=sh -c "nohup sudo -E '$APP_PATH'"
Path=$APP_DIR
Icon=network-vpn
Terminal=true
Categories=Network;Utility;
EOF
echo "✅ 快捷方式已创建到应用菜单。"

# 5. 复制快捷方式到桌面
DESKTOP_PATH="$HOME/Desktop"
if [ -d "$DESKTOP_PATH" ]; then
    cp "$APP_MENU_PATH/$DESKTOP_FILENAME" "$DESKTOP_PATH/"
    chmod +x "$DESKTOP_PATH/$DESKTOP_FILENAME"
    echo "✅ 已复制快捷方式到桌面。"
    # --- 关键步骤：使用 gio 设置元数据 ---
    # 检查 gio 命令是否存在
    if command -v gio &> /dev/null; then
        gio set "$DESKTOP_PATH/$DESKTOP_FILENAME" "metadata::trusted" yes
        echo "✅ 已自动将桌面快捷方式设置为'允许启动'。"
    else
        echo "⚠️  警告: 'gio' 命令未找到。您可能需要手动右键点击桌面图标并选择 '允许启动'。"
    fi
else
    echo "⚠️  警告：找不到桌面目录 '$DESKTOP_PATH'，已跳过。"
fi

# 6. 完成
echo ""
echo "🎉 全自动安装成功！"
echo ""
echo "现在，请按以下步骤操作："
echo "  1. 在桌面上找到 '$APP_NAME (Admin)' 图标。"
echo "  2. (如果需要) 右键点击它，选择 '允许启动 (Allow Launching)'。"
echo "  3. 双击图标，会在一个新的终端窗口里提示您输入密码。"
echo "  4. 输入密码后您的程序将会启动，之后您可以手动关闭该终端窗口。"